﻿namespace Microsoft.Samples.PlanMyNight.AddIns.EmailItinerary.Controllers
{
    using System.Collections.Generic;
    using System.ComponentModel.Composition;
    using System.Configuration;
    using System.Globalization;
    using System.Linq;
    using System.Net.Mail;
    using System.Text;
    using System.Web.Mvc;
    using Microsoft.Samples.PlanMyNight.AddIns.EmailItinerary.Notifications;
    using Microsoft.Samples.PlanMyNight.AddIns.EmailItinerary.Properties;
    using Microsoft.Samples.PlanMyNight.Data;
    using Microsoft.Samples.PlanMyNight.Entities;
    using Microsoft.Samples.PlanMyNight.Infrastructure;
    using Microsoft.Samples.PlanMyNight.Infrastructure.Mvc;

    [Export("EmailItinerary", typeof(IController))]
    [PartCreationPolicy(CreationPolicy.NonShared)]
    [OutputCache(NoStore = true, Duration = 0, VaryByParam = "*")]
    [HandleErrorWithContentType]
    public class EmailItineraryController : Controller
    {
        private readonly IItineraryContainer itineraryContainer;
        private readonly IItinerariesRepository itinerariesRepository;
        private readonly IActivitiesRepository activitiesRepository;
        private readonly INotificationService notifier;

        private const string ActivityTemplate =
@"{0}
{1} | {2} {3} | {4}
------------------------------------

";

        [ImportingConstructor]
        public EmailItineraryController(IServiceFactory serviceFactory) :
            this(
                serviceFactory.GetItineraryContainerInstance(),
                serviceFactory.GetItinerariesRepositoryInstance(),
                serviceFactory.GetActivitiesRepositoryInstance(),
                new NotificationService(
                    new Mailer(),
                    new ResourceTemplateReader(Resources.ResourceManager)))
        {
        }

        public EmailItineraryController(IItineraryContainer itineraryContainer, IItinerariesRepository itinerariesRepository, IActivitiesRepository activitiesRepository, INotificationService notifier)
        {
            this.itineraryContainer = itineraryContainer;
            this.itinerariesRepository = itinerariesRepository;
            this.activitiesRepository = activitiesRepository;
            this.notifier = notifier;
        }

        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Email(long? id)
        {
            if (id.HasValue)
            {
                ViewData["id"] = id.Value;
            }

            return View("Email");
        }

        [AcceptVerbs(HttpVerbs.Post)]
        [ValidateAntiForgeryToken()]
        public ActionResult Email(string email, long? id)
        {
            Itinerary itinerary = null;
            if (id.HasValue)
            {
                itinerary = this.itinerariesRepository.Retrieve(id.Value);
                if (itinerary != null && (!itinerary.IsPublic && itinerary.UserId == this.User.UserId()))
                {
                    itinerary = null;
                }
            }
            else
            {
                itinerary = this.itineraryContainer.GetCurrent();
            }

            if (itinerary != null && itinerary.Activities.Count > 0)
            {
                this.activitiesRepository.PopulateItineraryActivities(itinerary);

                var fields = new Dictionary<string, string>();

                var list = new StringBuilder();
                foreach (var activity in itinerary.Activities.OrderBy(a => a.Order).Select(a => a.Activity))
                {
                    list.Append(string.Format(CultureInfo.InvariantCulture, ActivityTemplate, activity.Name, activity.Street, activity.City, activity.Zip, activity.PhoneNumber));
                }

                fields.Add("list", list.ToString());
                fields.Add("name", itinerary.Name);
                fields.Add("description", itinerary.Description);

                var fromAddress = new MailAddress(ConfigurationManager.AppSettings["Email.FromAddress"], ConfigurationManager.AppSettings["Email.FromName"]);
                var destinyAddress = new MailAddress(email);

                this.notifier.SendMail("ItineraryTemplate", fromAddress, destinyAddress, fields);

                if (this.IsAjaxCall())
                {
                    return new JsonResult { JsonRequestBehavior = JsonRequestBehavior.AllowGet, Data = new { Status = "Ok" } };
                }
                else
                {
                    return RedirectToAction("EmailSuccess", new { area = RouteTableConfiguration.AreaName });
                }
            }

            return new EmptyResult();
        }

        public ActionResult EmailSuccess()
        {
            return View("EmailSuccess");
        }

        public PartialViewResult GetEmailToolbox(long? id)
        {
            if (id.HasValue)
            {
                ViewData["id"] = id.Value;
            }

            return PartialView("EmailToolbox");
        }
    }
}
